/*
 * Decompiled with CFR 0.152.
 */
package org.entrystore.repository.backup;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.file.Files;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import org.apache.commons.io.FileUtils;
import org.eclipse.rdf4j.rio.RDFFormat;
import org.eclipse.rdf4j.sail.SailException;
import org.entrystore.impl.RepositoryManagerImpl;
import org.entrystore.repository.config.Settings;
import org.entrystore.repository.util.DateUtils;
import org.entrystore.repository.util.FileOperations;
import org.quartz.InterruptableJob;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackupJob
implements Job,
InterruptableJob {
    private static final Logger log = LoggerFactory.getLogger(BackupJob.class);
    private static boolean interrupted = false;

    private static void writeErrorStatus(File errorFile, List<String> errors, String backupDateTime) {
        String errorFileContent = backupDateTime + "\n" + String.join((CharSequence)"\n", errors);
        try {
            FileOperations.writeStringToFile(errorFile, errorFileContent);
        }
        catch (IOException e) {
            log.error(e.getMessage());
        }
    }

    private static File getErrorStatusFile(File backupDirectory) {
        return new File(backupDirectory, "BACKUP_FAILED");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void execute(JobExecutionContext context) {
        block11: {
            if (interrupted) {
                throw new RuntimeException("Backup job \"execute()\" was interrupted");
            }
            try {
                JobDataMap dataMap = context.getJobDetail().getJobDataMap();
                RepositoryManagerImpl rm = (RepositoryManagerImpl)dataMap.get((Object)"rm");
                URI realURI = rm.getPrincipalManager().getAuthenticatedUserURI();
                try {
                    rm.getPrincipalManager().setAuthenticatedUserURI(rm.getPrincipalManager().getAdminUser().getURI());
                    rm.setModificationLockOut(true);
                    BackupJob.runBackup(context);
                }
                finally {
                    rm.setModificationLockOut(false);
                    rm.getPrincipalManager().setAuthenticatedUserURI(realURI);
                }
                boolean maintenance = (Boolean)dataMap.get((Object)"maintenance");
                if (maintenance) {
                    try {
                        rm.getPrincipalManager().setAuthenticatedUserURI(rm.getPrincipalManager().getAdminUser().getURI());
                        BackupJob.runBackupMaintenance(context);
                        break block11;
                    }
                    finally {
                        rm.getPrincipalManager().setAuthenticatedUserURI(realURI);
                    }
                }
                log.info("Backup maintenance not active");
            }
            catch (Exception e) {
                log.error(e.getMessage());
            }
        }
    }

    public static synchronized void runBackup(JobExecutionContext jobContext) {
        if (interrupted) {
            throw new RuntimeException("Backup job \"runBackup()\" was interrupted");
        }
        long beforeTotal = System.currentTimeMillis();
        log.info("Starting backup job");
        String currentDateTime = new SimpleDateFormat("yyyyMMddHHmmss").format(new Date());
        JobDataMap dataMap = jobContext.getJobDetail().getJobDataMap();
        RepositoryManagerImpl rm = (RepositoryManagerImpl)dataMap.get((Object)"rm");
        boolean gzip = dataMap.getBoolean("gzip");
        boolean includeFiles = dataMap.getBoolean("includeFiles");
        RDFFormat format = (RDFFormat)dataMap.getOrDefault((Object)"format", (Object)RDFFormat.TRIX);
        log.info("Backup gzip: {}", (Object)gzip);
        String exportPath = rm.getConfiguration().getString(Settings.BACKUP_FOLDER);
        if (exportPath == null) {
            log.error("Unknown backup path, please check the following setting: {}", (Object)Settings.BACKUP_FOLDER);
        } else {
            boolean simple = dataMap.getBoolean("simple");
            boolean deleteAfter = dataMap.getBoolean("deleteAfter");
            File oldBackupDirectory = new File(exportPath, "all-old");
            File newBackupDirectory = new File(exportPath, currentDateTime);
            LinkedList<String> errors = new LinkedList<String>();
            try {
                if (simple && (newBackupDirectory = new File(exportPath, "all")).isDirectory()) {
                    if (oldBackupDirectory.exists()) {
                        try {
                            FileUtils.deleteDirectory((File)oldBackupDirectory);
                            log.info("Deleted {}", (Object)oldBackupDirectory);
                        }
                        catch (IOException e) {
                            log.warn("Unable to delete {}: {}", (Object)oldBackupDirectory, (Object)e);
                        }
                    }
                    if (deleteAfter) {
                        if (newBackupDirectory.renameTo(oldBackupDirectory)) {
                            log.info("Renamed {} to {}", (Object)newBackupDirectory, (Object)oldBackupDirectory);
                        } else {
                            log.warn("Unable to rename {} to {}", (Object)newBackupDirectory, (Object)oldBackupDirectory);
                        }
                    } else {
                        try {
                            FileUtils.deleteDirectory((File)newBackupDirectory);
                            log.info("Deleted {}", (Object)newBackupDirectory);
                        }
                        catch (IOException e) {
                            log.warn("Unable to delete {}: {}", (Object)newBackupDirectory, (Object)e);
                        }
                    }
                }
                if (!newBackupDirectory.exists()) {
                    if (newBackupDirectory.mkdirs()) {
                        log.info("Created directory {}", (Object)newBackupDirectory);
                    } else {
                        log.error("Unable to create directory {}", (Object)newBackupDirectory);
                        log.info("Aborting backup due to unrecoverable error");
                        return;
                    }
                }
                long beforeMainExport = System.currentTimeMillis();
                log.info("Exporting main repository");
                String mainRepoFile = "repository." + format.getDefaultFileExtension() + (gzip ? ".gz" : "");
                try {
                    rm.exportToFile(rm.getRepository(), new File(newBackupDirectory, mainRepoFile).toURI(), gzip, format);
                    log.info("Exporting main repository took {} ms", (Object)(System.currentTimeMillis() - beforeMainExport));
                }
                catch (SailException se) {
                    log.error("Unable to export main repository {}", (Object)se.getMessage());
                    errors.add(se.getMessage());
                }
                if (rm.getProvenanceRepository() != null) {
                    long beforeProvExport = System.currentTimeMillis();
                    log.info("Exporting provenance repository");
                    String provRepoFile = "repository_prov." + format.getDefaultFileExtension() + (gzip ? ".gz" : "");
                    try {
                        rm.exportToFile(rm.getProvenanceRepository(), new File(newBackupDirectory, provRepoFile).toURI(), gzip, format);
                        log.info("Exporting provenance repository took {} ms", (Object)(System.currentTimeMillis() - beforeProvExport));
                    }
                    catch (SailException se) {
                        log.error("Unable to export provenance repository {}", (Object)se.getMessage());
                        errors.add(se.getMessage());
                    }
                } else {
                    log.info("Provenance repository is not configured and is therefore not be included in the backup");
                }
                if (includeFiles) {
                    String dataPath = rm.getConfiguration().getString(Settings.DATA_FOLDER);
                    if (dataPath == null) {
                        log.error("Unknown data path, please check the following setting: {}", (Object)Settings.DATA_FOLDER);
                    } else {
                        long beforeFileExport = System.currentTimeMillis();
                        File dataPathFile = new File(dataPath);
                        log.info("Copying data folder from {} to {}", (Object)dataPathFile, (Object)newBackupDirectory);
                        try {
                            FileOperations.copyPath(dataPathFile.toPath(), newBackupDirectory.toPath());
                            log.info("Copying data folder took {} ms", (Object)(System.currentTimeMillis() - beforeFileExport));
                        }
                        catch (IOException ioe) {
                            log.error("Unable to copy data folder from {} to {}", (Object)dataPathFile, (Object)newBackupDirectory);
                            errors.add(ioe.getMessage());
                        }
                    }
                } else {
                    log.warn("Files not included in backup due to configuration");
                }
                if (simple && deleteAfter && oldBackupDirectory.exists()) {
                    try {
                        FileUtils.deleteDirectory((File)oldBackupDirectory);
                        log.info("Deleted {}", (Object)oldBackupDirectory);
                    }
                    catch (IOException e) {
                        log.warn("Unable to delete {}: {}", (Object)oldBackupDirectory, (Object)e.getMessage());
                    }
                }
                try {
                    FileOperations.writeStringToFile(new File(newBackupDirectory, "BACKUP_DATE"), currentDateTime);
                }
                catch (IOException e) {
                    log.error(e.getMessage());
                }
                try {
                    Files.deleteIfExists(BackupJob.getErrorStatusFile(newBackupDirectory).toPath());
                }
                catch (IOException e) {
                    log.warn("Unable to delete {}: {}", (Object)BackupJob.getErrorStatusFile(newBackupDirectory), (Object)e.getMessage());
                }
            }
            catch (Exception e) {
                log.error("Backup failed: {}", (Object)e.getMessage(), (Object)e);
                errors.add(e.getMessage());
            }
            if (!errors.isEmpty()) {
                File errorStatusFile = BackupJob.getErrorStatusFile(newBackupDirectory);
                BackupJob.writeErrorStatus(errorStatusFile, errors, currentDateTime);
            }
        }
        log.info("Backup job done with execution, took {} ms in total", (Object)(System.currentTimeMillis() - beforeTotal));
    }

    public static synchronized void runBackupMaintenance(JobExecutionContext jobContext) {
        if (interrupted) {
            throw new RuntimeException("Backup job \"runBackupMaintenance()\" was interrupted");
        }
        log.info("Starting backup maintenance job");
        JobDataMap dataMap = jobContext.getJobDetail().getJobDataMap();
        RepositoryManagerImpl rm = (RepositoryManagerImpl)dataMap.get((Object)"rm");
        int upperLimit = (Integer)dataMap.get((Object)"upperLimit");
        int lowerLimit = (Integer)dataMap.get((Object)"lowerLimit");
        int expiresAfterDays = (Integer)dataMap.get((Object)"expiresAfterDays");
        log.info("upperlimit: {}, lowerLimit: {}, expiresAfterDays: {}", new Object[]{upperLimit, lowerLimit, expiresAfterDays});
        String exportPath = rm.getConfiguration().getString(Settings.BACKUP_FOLDER);
        Date today = new Date();
        if (exportPath == null) {
            log.error("Unknown backup path, please check the following setting: {}", (Object)Settings.BACKUP_FOLDER);
        } else {
            File backupFolder = new File(exportPath);
            if (!backupFolder.exists() || !backupFolder.isDirectory() || backupFolder.listFiles() == null) {
                log.error("Backup folder {} does not exist or is not a directory", (Object)exportPath);
            } else {
                Date oldestDate;
                SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMddHHmmss");
                LinkedList<Date> backupFolders = new LinkedList<Date>();
                for (File file : backupFolder.listFiles()) {
                    if (!file.isDirectory() || file.isHidden()) {
                        log.info("Ignoring: {}", (Object)file);
                        continue;
                    }
                    try {
                        Date date = formatter.parse(file.getName());
                        backupFolders.add(date);
                        log.info("Found backup folder: {}", (Object)file);
                    }
                    catch (ParseException pe) {
                        log.warn("Skipping path: {}. Reason: {}", (Object)file, (Object)pe.getMessage());
                    }
                }
                if (backupFolders.size() <= lowerLimit) {
                    log.info("Lower limit not reached - backup maintenance job done with execution");
                    return;
                }
                backupFolders.sort((a, b) -> {
                    if (a.after((Date)b)) {
                        return 1;
                    }
                    if (a.before((Date)b)) {
                        return -1;
                    }
                    return 0;
                });
                if (backupFolders.size() > upperLimit) {
                    int nrRemoveItems = backupFolders.size() - upperLimit;
                    log.info("Upper limit is {}, will delete {} backup folder(s)", (Object)upperLimit, (Object)nrRemoveItems);
                    for (int i = 0; i < nrRemoveItems; ++i) {
                        String folder = formatter.format((Date)backupFolders.get(i));
                        File f = new File(exportPath, folder);
                        if (FileOperations.deleteDirectory(f)) {
                            backupFolders.remove(i);
                            log.info("Deleted {}", (Object)f);
                            continue;
                        }
                        log.info("Unable to delete {}", (Object)f);
                    }
                }
                if (DateUtils.daysBetween(oldestDate = (Date)backupFolders.getFirst(), today) > (long)expiresAfterDays) {
                    Date d;
                    int i = 0;
                    for (int size = backupFolders.size(); lowerLimit < size && DateUtils.daysBetween(d = (Date)backupFolders.get(i), today) > (long)expiresAfterDays; --size) {
                        String folder = formatter.format((Date)backupFolders.get(i));
                        File f = new File(exportPath, folder);
                        if (FileOperations.deleteDirectory(f)) {
                            backupFolders.remove(i);
                            log.info("Deleted {}", (Object)f);
                            continue;
                        }
                        log.info("Unable to delete {}", (Object)f);
                    }
                }
            }
        }
        log.info("Backup maintenance job done with execution");
    }

    public void interrupt() {
        interrupted = true;
    }
}

