/*
 * Decompiled with CFR 0.152.
 */
package org.entrystore.rest.auth;

import java.net.URI;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import org.entrystore.Entry;
import org.entrystore.PrincipalManager;
import org.entrystore.User;
import org.entrystore.config.Config;
import org.entrystore.repository.RepositoryManager;
import org.entrystore.repository.config.Settings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UserTempLockoutCache {
    private static final Logger log = LoggerFactory.getLogger(UserTempLockoutCache.class);
    private final int configAllowedFailedLoginAttempts;
    private final Duration configUserLockoutDuration;
    private final boolean configIncludeAdmin;
    private final PrincipalManager pm;
    private final ConcurrentMap<String, UserTemporaryLockout> userTempLockoutMap = new ConcurrentHashMap<String, UserTemporaryLockout>();

    public UserTempLockoutCache(RepositoryManager rm, PrincipalManager pm) {
        Config config = rm.getConfiguration();
        this.pm = pm;
        this.configAllowedFailedLoginAttempts = config.getInt(Settings.AUTH_TEMP_LOCKOUT_MAX_ATTEMPTS, 5);
        this.configUserLockoutDuration = config.getDuration(Settings.AUTH_TEMP_LOCKOUT_DURATION, Duration.ofMinutes(5L));
        this.configIncludeAdmin = config.getBoolean(Settings.AUTH_TEMP_LOCKOUT_ADMIN, true);
    }

    public void succeedLogin(String userName) {
        this.userTempLockoutMap.remove(userName);
    }

    public void failLogin(String userName) {
        if (this.configAllowedFailedLoginAttempts < 0 || this.configUserLockoutDuration.isZero()) {
            return;
        }
        Entry userEntry = this.pm.getPrincipalEntry(userName);
        if (userEntry == null) {
            log.warn("Login attempt failed, user does not exist: {}", (Object)userName);
            return;
        }
        URI userUri = userEntry.getResourceURI();
        User user = this.pm.getUser(userUri);
        log.info("User [{}] failed login attempt due to providing wrong password", (Object)userName);
        if (this.pm.isUserAdminOrAdminGroup(userUri) && !this.configIncludeAdmin) {
            log.warn("Login attempt of user [{}] is not counted towards temporary lockout because of configuration for admin users", (Object)userName);
            return;
        }
        UserTemporaryLockout lockoutEntry = this.userTempLockoutMap.getOrDefault(userName, new UserTemporaryLockout(user, 1, null));
        if (lockoutEntry.disableUntil() != null && LocalDateTime.now().isAfter(lockoutEntry.disableUntil())) {
            new UserTemporaryLockout(user, 1, null);
        }
        if (lockoutEntry.failedLogins() < this.configAllowedFailedLoginAttempts) {
            lockoutEntry = new UserTemporaryLockout(user, lockoutEntry.failedLogins() + 1, null);
        } else {
            LocalDateTime lockedOutUntil = LocalDateTime.now().plus(this.configUserLockoutDuration);
            lockoutEntry = new UserTemporaryLockout(user, lockoutEntry.failedLogins() + 1, lockedOutUntil);
            log.warn("User [{}] failed too many login attempts and will be locked out until {}", (Object)userName, (Object)lockedOutUntil);
        }
        this.userTempLockoutMap.put(userName, lockoutEntry);
        this.pm.setAuthenticatedUserURI(null);
    }

    public boolean userIsLockedOut(String userName) {
        return this.getLockedOutUser(userName) != null;
    }

    public UserTemporaryLockout getLockedOutUser(String userName) {
        UserTemporaryLockout lockedOutUser = (UserTemporaryLockout)this.userTempLockoutMap.get(userName);
        if (lockedOutUser == null || lockedOutUser.disableUntil() == null) {
            return null;
        }
        if (LocalDateTime.now().isAfter(lockedOutUser.disableUntil())) {
            log.info("User [{}] stopped being locked out", (Object)userName);
            this.userTempLockoutMap.remove(userName);
            return null;
        }
        return lockedOutUser;
    }

    public List<UserTemporaryLockout> getLockedOutUsers() {
        List<UserTemporaryLockout> lockedOutUsers = this.userTempLockoutMap.entrySet().stream().filter(failedLoginCount -> this.userIsLockedOut(((UserTemporaryLockout)failedLoginCount.getValue()).user().getName())).map(Map.Entry::getValue).collect(Collectors.toList());
        return lockedOutUsers;
    }

    public record UserTemporaryLockout(User user, int failedLogins, LocalDateTime disableUntil) {
    }
}

